﻿/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using Newtonsoft.Json;
using Ext.Net;
using System.IO;
using System.Data;
using DataAccess;
using System.Collections;
using System.Web.Script.Serialization;
using System.Xml;

//setup an alias for all direct method calls on this control
[DirectMethodProxyID(IDMode = DirectMethodProxyIDMode.Alias, Alias = "UCINSTRUMENTSEL")]

public partial class ucInstrumentSelection : System.Web.UI.UserControl
{
    public BaseMaster BaseMstr { get; set; }

    /// <summary>
    /// US:6678 patient pathway event id
    /// </summary>
    public string PatPWEventID
    {
        get
        {
            string strValue = "";
            if (Session["UCINSTRUMENTSEL_PATPWEVENTID"] != null)
            {
                strValue = Session["UCINSTRUMENTSEL_PATPWEVENTID"].ToString();
            }

            return strValue;
        }
        set { Session["UCINSTRUMENTSEL_PATPWEVENTID"] = Convert.ToString(value); }
    }

    /// <summary>
    /// US:6678 selected instrument ids
    /// </summary>
    public string SelectedMIDS
    {
        get
        {
            string strValue = "";
            if (Session["UCINSTRUMENTSEL_MIDS"] != null)
            {
                strValue = Session["UCINSTRUMENTSEL_MIDS"].ToString();
            }

            return strValue;
        }
        set { Session["UCINSTRUMENTSEL_MIDS"] = Convert.ToString(value); }
    }

    /// <summary>
    /// US:5859 US:6678 show the intrument selection/event update popup
    /// </summary>
    public void Show(long lPatPWEventID)
    {
        PatPWEventID = Convert.ToString(lPatPWEventID);
             
        //clear selected mids
        SelectedMIDS = String.Empty;
        
        //clear the combo selection
        cboInstrumentGroup.ClearValue();

        //clear the instruments
        gpInstruments.GetStore().RemoveAll();
        storInstruments.SetDataFromJson(CDataUtils2.GetEmptyJSON());

        //clear the selected instruments
        gpSelInstruments.GetStore().RemoveAll();
        storSelInstruments.SetDataFromJson(CDataUtils2.GetEmptyJSON());
               
        //load the event title and date
        CCPA cpa = new CCPA();
        DataSet dsEvent = cpa.GetPatPWEventByIDDS(BaseMstr, BaseMstr.SelectedPatientID, lPatPWEventID);
        tfEPWEventTitle.Text = CDataUtils2.GetDSStringValue(dsEvent, "EVENT_TITLE");
        dfEPWEventDate.SelectedDate = CDataUtils2.GetDSDateTimeValue(dsEvent, "DATE_SCHEDULED");
 
        //load the selected instruments from the current list of modules
        DataSet dsModules = cpa.GetPatPWEventModuleDS(BaseMstr, 
                                                      BaseMstr.SelectedPatientID, 
                                                      lPatPWEventID);

        //remove records from the dataset that are completed
        foreach (DataTable table in dsModules.Tables)
        {
            foreach (DataRow row in table.Rows)
            {
                if (CDataUtils2.GetDSLongValue(row, "intake_id") > 0)
                {
                    row.Delete();
                }
                else
                {
                    //module data is not encoded in the db...
                    row["MODULE"] = HttpUtility.HtmlEncode(row["MODULE"].ToString());
                    row["DESCRIPTION"] = HttpUtility.HtmlEncode(row["DESCRIPTION"].ToString());
                    row["MODULE_ALT_LANG"] = HttpUtility.HtmlEncode(row["MODULE_ALT_LANG"].ToString());
                }
      
            }
        }
        dsModules.AcceptChanges();
        
        string strMIDS = "";

        foreach (DataTable dt in dsModules.Tables)
        {
            foreach (DataRow dr in dt.Rows)
            {
                //get instrument info
                string strMID = CDataUtils2.GetDSStringValue(dr, "MID");
                string strModule = HttpUtility.HtmlDecode(CDataUtils2.GetDSStringValue(dr, "MODULE"));
                string strDescription = HttpUtility.HtmlDecode(CDataUtils2.GetDSStringValue(dr, "DESCRIPTION"));

                //cache the selected mid
                strMIDS += "," + strMID + ",";
                strMIDS = strMIDS.Replace(",,", ",");
                SelectedMIDS = strMIDS;

                CDataConverter cdc = new CDataConverter();
                string strJsonRecord = cdc.GetJsonDRString(dr);

                //add instrument to second list 
                //turn the json record passed to us into and object
                JavaScriptSerializer serializer = new JavaScriptSerializer();
                Dictionary<string, object> ds = serializer.Deserialize<Dictionary<string, object>>(strJsonRecord);

                //add the object to the grid's store
                storSelInstruments.Add(ds);

                //split mids from cache to work with, only include mids 
                //that are not from the currently selected group
                RowSelectionModel sm = gpSelInstruments.GetSelectionModel() as RowSelectionModel;
                if (sm != null)
                {
                    sm.SelectedRows.Add(new SelectedRow(strMID));

                    //if you change selected rows in a 
                    //direct method you must call update selection!
                    sm.UpdateSelection();
                }
            }
        }
                
        wndAddInstrument.Show();
    }

    /// <summary>
    /// US:5859 US:6678  initialize the instrument selection/event update dialog
    /// </summary>
    /// <returns></returns>
    public bool InitializeControl()
    {
        //clear selected mids
        SelectedMIDS = String.Empty;
        
        //clear the combo
        cboInstrumentGroup.GetStore().RemoveAll();
        cboInstrumentGroup.Items.Clear();

        //clear the instruments
        gpInstruments.GetStore().RemoveAll();
        gpInstruments.Items.Clear();
        storInstruments.SetDataFromJson(CDataUtils2.GetEmptyJSON());

        //clear the selection
        RowSelectionModel sm = gpInstruments.GetSelectionModel() as RowSelectionModel;
        if (sm != null)
        {
            sm.SelectedRow = null;
        }
        
        //load group type radio list
        CIntake im = new CIntake();
        DataSet dsGroups = im.GetModuleGroupTypeDS(BaseMstr);
        CExtUtils.LoadExtRadioGroup(rgInstrumentGroupType,
                                    dsGroups,
                                    "module_group_type_id",
                                    "module_group_descr");

        //select the first radio by default
        int i = 0;
        foreach (Ext.Net.Radio r in rgInstrumentGroupType.Items)
        {
            string strVal = r.Value.ToString();

            //check the first radio button, check and fire the load
            if (i == 0)
            {
                r.Checked = true;
                LoadInstrumentGroups(CDataUtils2.ToLong(strVal));
                break;
            }
            i++;
        }

        return true;
    }

    /// <summary>
    /// US:5859 US:6678 instrument selection/event update control page load
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void Page_Load(object sender, EventArgs e)
    {
        if (!IsPostBack)
        {
            if (!X.IsAjaxRequest)
            {
                
            }

        }
    }

    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    ///US:5859 US:6678  load the instrument grid based on the combo group type
    /// </summary>
    /// <param name="strGroupTypeID"></param>
    public void OnGroupTypeSelect(string strGroupTypeID)
    {
        //cache selected instruments
        //CacheSelectedInstruments();

        //get the id of the radio button
        //example:
        //{"App.rgInstrumentGroupType_Group":"App.rgInstrumentGroupType_200"}
        //If I find a better way I will implement it
        string strFind = ":\"App.rgInstrumentGroupType_";
        int nPos = strGroupTypeID.IndexOf(strFind);
        if (nPos == -1)
        {
            return;
        }
        string strID = strGroupTypeID.Substring(nPos + strFind.Length);
        nPos = strID.IndexOf("\"");
        strID = strID.Substring(0, nPos);
        long lGroupTypeID = CDataUtils2.ToLong(strID);

        //clear the combo
        cboInstrumentGroup.GetStore().RemoveAll();
        cboInstrumentGroup.Items.Clear();

        //clear the instruments
        gpInstruments.GetStore().RemoveAll();
        storInstruments.SetDataFromJson(CDataUtils2.GetEmptyJSON());

        //clear the selection
        RowSelectionModel sm = gpInstruments.GetSelectionModel() as RowSelectionModel;
        if (sm != null)
        {
            sm.SelectedRow = null;
            sm.SelectedIndex = -1;

            sm.ClearSelection();
            sm.UpdateSelection();
        }

        //load the groups into the cbo
        LoadInstrumentGroups(lGroupTypeID);
    }


    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    ///US:5859 US:6678  fires when a group is selected from the combo
    /// </summary>
    public void OnGroupSelect()
    {
        //clear the store
        storInstruments.SetDataFromJson(CDataUtils2.GetEmptyJSON());
        if (cboInstrumentGroup.SelectedItem == null)
        {
            return;
        }

        //get the group id from the combo
        string strGroupID = cboInstrumentGroup.SelectedItem.Value;
        
        //load the instruments for this group
        LoadInstruments();
    }

    /// <summary>
    /// us:5859 US:6678 load instruments gridpanel by group id
    /// </summary>
    protected void LoadInstruments()
    {
        string strGroupID = cboInstrumentGroup.SelectedItem.Value;

        //get the modules ds
        CIntake im = new CIntake();
        DataSet dsMID = im.GetModuleGroupMIDDS(BaseMstr, CDataUtils2.ToLong(strGroupID));
        if (CDataUtils2.IsEmpty(dsMID))
        {
            return;
        }
        foreach (DataTable table in dsMID.Tables)
        {
            foreach (DataRow row in table.Rows)
            {
                if (CDataUtils2.GetDSLongValue(row, "intake_id") > 0)
                {
                    row.Delete();
                }
                else
                {
                    //module data is not encoded in the db...
                    row["MODULE"] = HttpUtility.HtmlEncode(row["MODULE"].ToString());
                    row["DESCRIPTION"] = HttpUtility.HtmlEncode(row["DESCRIPTION"].ToString());
                    row["MODULE_ALT_LANG"] = HttpUtility.HtmlEncode(row["MODULE_ALT_LANG"].ToString());
                }
            }
        }
        //clear the store
        storInstruments.SetDataFromJson(CDataUtils2.GetEmptyJSON());

        //load the store from the ds
        CDataConverter cdc = new CDataConverter();
        storInstruments.SetDataFromJson(cdc.GetJsonDSString(dsMID));

        //now that the list is loaded, check the items that were previously cached
        string strCachedMIDs = SelectedMIDS;
       
        RowSelectionModel sm = gpInstruments.GetSelectionModel() as RowSelectionModel;
        if (sm != null)
        {
            //clear all current selections
            sm.SelectedRows.Clear();

            //reselect all items in the selected list
            string[] splitText = strCachedMIDs.Split(new Char[] { ',' });
            foreach (string strMID in splitText)
            {
                if (!String.IsNullOrEmpty(strMID))
                {
                    sm.SelectedRows.Add(new SelectedRow(strMID));
                }
            }
            //if you change selected rows in a 
            //direct method you must call update selection!
            sm.UpdateSelection();
        }
    }

    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    ///US:5859 US:6678  user checked an instrument
    /// </summary>
    public void OnInstrumentSel(string strJsonRecord)
    {

        //get instrument info
        string strMID = CDataUtils2.GetJsonRecordValue("MID", strJsonRecord);
        string strModule = HttpUtility.HtmlEncode(CDataUtils2.GetJsonRecordValue("MODULE", strJsonRecord));
        string strDescription = HttpUtility.HtmlEncode(CDataUtils2.GetJsonRecordValue("DESCRIPTION", strJsonRecord));

        //cache the selected mid
        string strMIDS = SelectedMIDS;
        //only add if its not already in the list
        if (strMIDS.IndexOf("," + strMID + ",") != -1)
        {
            return;
        }
        strMIDS += "," + strMID + ",";
        strMIDS = strMIDS.Replace(",,", ",");
        SelectedMIDS = strMIDS;


        //add instrument to second list 
        //turn the json record passed to us into and object
        JavaScriptSerializer serializer = new JavaScriptSerializer();
        Dictionary<string, object> ds = serializer.Deserialize<Dictionary<string, object>>(strJsonRecord);

        if (ds != null)
        {
            //add the object to the grid's store
            storSelInstruments.Add(ds);
        }

        //split mids from cache to work with, only include mids 
        //that are not from the currently selected group
        RowSelectionModel sm = gpSelInstruments.GetSelectionModel() as RowSelectionModel;
        if (sm != null)
        {
            sm.SelectedRows.Add(new SelectedRow(strMID));

            //if you change selected rows in a 
            //direct method you must call update selection!
            sm.UpdateSelection();
        }

    }

    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    ///US:5859 US:6678 user unchecked an instrument
    /// </summary>
    public void OnInstrumentDeSel(string strJsonRecord)
    {

        //get instrument info
        string strMID = CDataUtils2.GetJsonRecordValue("MID", strJsonRecord);
        string strModule = HttpUtility.HtmlEncode(CDataUtils2.GetJsonRecordValue("MODULE", strJsonRecord));
        string strDescription = HttpUtility.HtmlEncode(CDataUtils2.GetJsonRecordValue("DESCRIPTION", strJsonRecord));

        //remove instrument from second list

        //turn the json record passed to us into and object
        JavaScriptSerializer serializer = new JavaScriptSerializer();
        Dictionary<string, object> ds = serializer.Deserialize<Dictionary<string, object>>(strJsonRecord);
        if (ds != null)
        {
            //remove the item from the grid
            storSelInstruments.Remove(ds["MID"]);

            //remove the selected mid from the cache
            string strMIDS = SelectedMIDS;
            strMIDS = strMIDS.Replace("," + strMID + ",", ",,");
            strMIDS = strMIDS.Replace(",,", ",");
            SelectedMIDS = strMIDS;
        }

    }


    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    ///US:5859 US:6678 user unchecked an instrument
    /// </summary>
    public void OnRemoveInstrument()
    {
        RowSelectionModel sm = gpSelInstruments.GetSelectionModel() as RowSelectionModel;
        if (sm != null)
        {
            if (sm.SelectedRow == null)
            {
                BaseMstr.ShowSystemFeedback("Please select an instrument to remove!", "Selection Error");
                return;
            }

            string strJsonRecord = "{\"MID\":" + sm.SelectedRow.RecordID + "}";

            //turn the json record passed to us into and object
            JavaScriptSerializer serializer = new JavaScriptSerializer();
            Dictionary<string, object> ds = serializer.Deserialize<Dictionary<string, object>>(strJsonRecord);

            //remove the item from the grid
            storSelInstruments.Remove(ds["MID"]);

            //remove the selected mid from the cache
            string strMIDS = SelectedMIDS;
            strMIDS = strMIDS.Replace("," + sm.SelectedRow.RecordID + ",", ",,");
            strMIDS = strMIDS.Replace(",,", ",");
            SelectedMIDS = strMIDS;

            LoadInstruments();
        }
    }


    /// <summary>
    ///US:5859 US:6678 load instrument groups into the combo
    /// </summary>
    /// <param name="lGroupTypeID"></param>
    public void LoadInstrumentGroups(long lGroupTypeID)
    {
        //clear value
        cboInstrumentGroup.SetValue(null);

        //clear the store
        storModuleGroups.SetDataFromJson(CDataUtils2.GetEmptyJSON());

        //get the group types ds
        CIntake im = new CIntake();
        DataSet dsModuleGroup = im.GetModuleGroupDS(BaseMstr, lGroupTypeID);
        if (CDataUtils2.IsEmpty(dsModuleGroup))
        {
            return;
        }

        //load the store from the ds
        CDataConverter cdc = new CDataConverter();
        storModuleGroups.SetDataFromJson(cdc.GetJsonDSString(dsModuleGroup));
    }

    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    ///US:5963 US:6678 ok button handler for instrument selection/event update popup
    /// </summary>
    protected void OnInstrSelOK(object sender, DirectEventArgs e)
    {
        //event title must be at least 1 chars dont have to have a title!
        //  if (tfEPWEventTitle.Text.Length < 1)
        //  {
        //      BaseMstr.ShowSystemFeedback("Please enter a valid event title!", "Data Entry Error");
        //      return -1;
        //  }

        //must have a date selected
        DateTime dtSchedule = dfEPWEventDate.SelectedDate;
        if (dtSchedule.Year <= 1)
        {
            BaseMstr.ShowSystemFeedback("Please select a valid date!", "Data Entry Error");
            return;
        }

        //JSON representation
        string gpJSON = e.ExtraParams["gpSelParam"];
        gpJSON = HttpUtility.HtmlDecode(gpJSON);


        //XML representation
        XmlNode gpXml = JSON.DeserializeXmlNode("{records:{record:" + gpJSON + "}}");

        //array of Dictionaries
        Dictionary<string, string>[] gpData = JSON.Deserialize<Dictionary<string, string>[]>(gpJSON);

        //all selected modules much have module for set
        for (int i = 0; i < gpData.Count(); i++)
        {
            if (String.IsNullOrEmpty(gpData[i]["MODULE_FOR"]))
            {
                BaseMstr.ShowSystemFeedback("Please choose who the instrument is for: patient, provider or both for all selected instruments!", "Data Entry Error");
                return;
            }
        }

        //get mid and module_for values
        string strFors = String.Empty;
        string strMids = String.Empty;
        for (int i = 0; i < gpData.Count(); i++)
        {
            strFors += gpData[i]["MODULE_FOR"];
            if (i != gpData.Count() - 1)
            {
                strFors += ",";
            }

            strMids += gpData[i]["MID"];
            if (i != gpData.Count() - 1)
            {
                strMids += ",";
            }
        }

        //should we adjust future dates?
        long lAdjustFuture = 0;
        if (chkEPWAdjustFuture.Checked)
        {
            lAdjustFuture = 1;
        }

        //update event info and the list of modules. 
        //this will remove all current modules and replace with 
        //only the selected modules...
        CCPA cpa = new CCPA();
        cpa.UpdatePWEventModules(BaseMstr,
                                 BaseMstr.SelectedPatientID,
                                 CDataUtils2.ToLong(PatPWEventID),
                                 tfEPWEventTitle.Text,
                                 dfEPWEventDate.SelectedDate,
                                 lAdjustFuture,
                                 strMids,
                                 strFors);


        //close the add instrument popup
        wndAddInstrument.Close();
    }
    
    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>", SuccessFn = "fnReloadEventsNodes")]
    /// <summary>
    /// Reload the events nodes
    /// </summary>
    /// <param name="?"></param>
    /// <returns></returns>
    public long ReloadPWEventNodes()
    {
        long lPWEventID = CDataUtils2.ToLong(PatPWEventID);
        return lPWEventID;
    }

    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    ///US:5963 US:6678 cancel button handler for instrument selection/event update popup
    /// </summary>
    public void OnInstrumentSelCancel()
    {
        wndAddInstrument.Close();
    }
}